#include "KMotionDef.h"
#include "CommonDefs.h"
#include "PCComm.h"

void MoveTo(int axis, double pos);
void WaitDone(int axis);
int ProbeAxis(int axis, int dir);
int DoProbe(int axis, int dir, double dist, double vel, double stepsperunit, double backoff);
int HomeAxis(int axis);
int DoHome(int axis, int dir, double fastdist, double fastvel, double slowdist, double slowvel, double stepsperunit);
int _DoHomeProbe(int axis, double dist, double vel, int pin, int polarity);
double GetAxisResolution(int axis);


float stepsperunit[4] = { 0, 0, 0, 0 };

/**********************************************************
 * ProbeAxis(int axis, int dir) - Perform a Probe operation
 * (similar to G31) on the specified axis, in the specified 
 *	direction.
 *
 *	axis - axis to move
 *	dir - direction to move, 1=Positive, -1=Negative
 *********************************************************/
int ProbeAxis(int axis, int dir)
{
	//printf("ProbeAxis(%d, %d)\n", axis, dir);

	// Make sure probe is not already triggered
	if (!ReadBit(PROBE_PIN))
	{
		MsgBox("Probe active! Please correct.    ", MB_OK);
		return;
	}

	double stepsperunit = GetAxisResolution(axis);

	// Do fast probe
	if (DoProbe(axis, dir, PROBE_DISTANCE, FAST_PROBE_SPEED, stepsperunit, PROBE_BACK_OFF))
	{
		MsgBox("Fast probe failed!    ", MB_OK);
		return;
	}
	
	// Move back to 0.010" short of trigger position
	Zero(axis);
	MoveTo(axis, (PROBE_BACK_OFF - 0.010) * stepsperunit * dir);

	// Do slow probe
	if (DoProbe(axis, dir, 0.020, SLOW_PROBE_SPEED, stepsperunit, PROBE_BACK_OFF))
	{
		MsgBox("Slow probe failed!    ", MB_OK);
		return;
	}

	MsgBox("Remove Touchplate.    ", MB_OK);

	MoveTo(axis, (PROBE_BACK_OFF + TOUCHPLATE_THICKNESS) * stepsperunit * dir);
	// Zero the axis here
	Zero(axis);

	DoPCFloat(PC_COMM_SET_X + axis, 0.0);
	MsgBox("Probe Done!  Remember To Set Offset.    ", MB_OK);
}

/**********************************************************
 * HomeAxis(int axis, int dir) - Perform a Home operation
 * on the specified axis, in the specified direction.
 *
 *	axis - axis to move
 *	dir - direction to move, 1=Positive, -1=Negative
 *********************************************************/
int HomeAxis(int axis, int dir)
{
	//printf("HomeAxis(%d, %d)\n", axis, dir);
	double stepsperunit = GetAxisResolution(axis);
	int cmd;
	double dist = 0.5;
	double pos = 0.0;

	switch (axis)
	{
		case AXIS_X:
		case AXIS_Y:
			break;

		case AXIS_Z:
			dist = 2.5;
			pos = 2.0;
			break;

		case AXIS_A:
			break;
	}
	if (DoHome(axis, dir, dist, FAST_HOME_SPEED, 0.2, SLOW_HOME_SPEED, stepsperunit))
	{
		MsgBox("Home failed!    ", MB_OK);
		return;
	}
	// Zero the axis here
	Zero(axis);

	DoPCFloat(PC_COMM_SET_X + axis, pos);
	MsgBox("Home Done!  Remember To Set Offset.    ", MB_OK);
}

/**********************************************************
 * DoProbe() - Perform a Probe (G31-like) Operation
 *
 *	This function performs a probe operation, first
 *  moving in the specified direction, then backing off
 *	the specified distance.
 *
 *	axis - axis to move
 *	dist - maximum distance for move, in inch/mm
 *	vel - move velocity, counts/sec
 *	stepsperunit - steps per unit (inch/mm)
 *	backoff - distance to backoff from probe trigger point,
 *			  in inch/mm
 *
 *	The specified axis is moved up to the specified distance
 *	at the specified velocity, stopping if the specified input
 *	goes active.  The axis is then moved back to exactly
 *	the the position where the probe triggered.  Finally, it
 *	is moved away the distance specified by backoff.
 *
 *	Returns 0 if successful, -1 if probe never triggers
 *********************************************************/
int DoProbe(int axis, int dir, double dist, double vel, double stepsperunit, double backoff)
{
	int ret = 0;

	//printf("DoProbe(axis=%d, dir=%d, dist=%.4f, vel=%.4f, spi=%.4f, backoff=%.4f)\n", axis, dir, dist, vel, stepsperunit, backoff); 
	double probedist = dist * stepsperunit * dir;
	double probevel = vel * stepsperunit / 60.0;
	ret = _DoHomeProbe(axis, probedist , probevel, PROBE_PIN, 0);
	if (ret == 0)
	{
		// Now back off
		double pos = chan[axis].Dest;
		pos += backoff * stepsperunit * dir * -1;
		MoveTo(axis, pos );
	}
	return ret;
}

/**********************************************************
 * DoHome() - Perform a Home (G28.1-like) Operation
 *
 *	This function performs a two-stop home operation, first
 *  moving fast in the specified direction, the slowly in the
 *	opposite direction, for greater accuracy.
 *
 *	axis - axis to move
 *	fastdist - maximum distance for fast move, in inch/mm
 *	fastvel - fast move velocity (counts/sec)
 *	slowdist - maximum distance for slow move, in inch/mm
 *	slowvel - slow move velocity (counts/sec)
 *	stepsperunit - steps per unit (inch/mm)
 *
 *	The specified axis is moved up to the specified distance
 *	at the specified velocity, stopping if the specified input
 *	goes low.  The axis is then moved back to exactly
 *	the the position where the probe triggered.  
 *
 *	Returns 0 if successful, -1 if probe never triggers
 *********************************************************/
int DoHome(int axis, int dir, double fastdist, double fastvel, double slowdist, double slowvel, double stepsperunit)
{
	int ret = 0;

	//printf("DoHome(axis=%d, dir=%d, fastdist=%.4f, fastvel=%.4f,  slowdist=%.4f, slowvel=%.4fspi=%.4f )\n", axis, dir, fastdist, fastvel, slowdist, slowvel, stepsperunit); 

	// Disable limits
	DisableAxis(AXIS_Z);
	int SaveZLimits = ch2->LimitSwitchOptions;
	ch2->LimitSwitchOptions = LIMIT_IGNORE;
	EnableAxisDest(AXIS_Z, ch2->Dest);

	double homedist = (double)(fastdist * stepsperunit * dir);
	double homevel = fastvel * stepsperunit / 60.0;
	ret = _DoHomeProbe(axis, homedist, homevel, HOME_PIN, 0);
	if (ret == 0)
	{
		// Now move back until switch opens
		homedist = (double)(slowdist * stepsperunit * dir * -1);
		homevel = slowvel * stepsperunit / 60.0;
		ret = _DoHomeProbe(axis, homedist, homevel, HOME_PIN, 1);
	}

Exit:
	// Re-enable limits
	DisableAxis(AXIS_Z);
	ch2->LimitSwitchOptions = SaveZLimits;
	EnableAxisDest(AXIS_Z, ch2->Dest);
	return ret;
}

/**********************************************************
 * _DoHomeProbe() - Primitive Home/Probe operation
 *
 *	axis - axis to move
 *	dist - maximum distance to move, in counts
 *	vel - move velocity, in counts/sec
 *	pin - input pin to monitor
 *	polarity - input is active high if 1
 *
 *	The specified axis is moved up to the specified distance
 *	at the specified velocity, stopping if the specified input
 *	goes active.  The axis is then moved back to exactly
 *	the position where the input went active.  
 *	If polarity is 1, then the input is active high.
 *
 *	Returns 0 if successful, -1 if probe never triggers
 *********************************************************/
int _DoHomeProbe(int axis, double dist, double vel, int pin, int polarity)
{
int ret = 0;
double triggerPos = 0.0;

	//printf("_DoHomeProbe(%d, %.4f, %.4f, %d, %d)\n", axis, (float)dist, (float)vel, pin, polarity);
	// Start moving
	MoveRelAtVel(axis, dist, vel);
	while (ReadBit(pin) ^ polarity) {
		if (CheckDone(axis))
		{
			// Ran distance, no probe trigger
			ret = -1;
			break;
		}
	}
	// Remember where probe triggered
	triggerPos = chan[axis].Dest;
	// Stop the axis
	Jog(axis, 0);
	WaitDone(axis);
	// Move back to trigger position
	MoveTo(axis, triggerPos);
	return ret;
}

/**********************************************************
 * GetAxisResolution(int axis) - Get resolution of specified
 * axis from PC.  Resolution is in count/inch or counts/mm
 *********************************************************/
double GetAxisResolution(int axis)
{
	if (stepsperunit[0] == 0)
	{
		DoPCInt(PC_COMM_GETAXISRES, 0);
		stepsperunit[0] = *(float *)&persist.UserData[PC_COMM_PERSIST+1];
		stepsperunit[1] = *(float *)&persist.UserData[PC_COMM_PERSIST+2];
		stepsperunit[2] = *(float *)&persist.UserData[PC_COMM_PERSIST+3];
		stepsperunit[3] = *(float *)&persist.UserData[PC_COMM_PERSIST+4];
		//printf("%f, %f, %f, %f\n", stepsperunit[0], stepsperunit[1], stepsperunit[2], stepsperunit[3]);
	}
	return (double)stepsperunit[axis];
}

/**********************************************************
 * MoveTo(axis, pos) - Move axis to specified position,
 * return when move complete
 *********************************************************/
void MoveTo(int axis, double pos)
{
	//printf("MoveTo(%d, %.1f)\n", axis, pos);
	MoveAtVel(axis, pos, chan[axis].Vel);
	WaitDone(axis);
}

/**********************************************************
 * WaitDone(axis) - Wait for axis not moving
 *********************************************************/
void WaitDone(int axis)
{
	while (!CheckDone(axis))
		;
}



