// TrajectoryPlanner.c  Tom Kerekes 09/11/02

#include "stdafx.h"

#ifndef TWO_PI
#define TWO_PI (2.0*3.141592653589793238)
#endif

#ifndef HALF_PI
#define HALF_PI (0.5*3.141592653589793238)
#endif

double BreakAngle = (30.0 * PI/180.0);  // amount of change in direction where we need to stop

int nsegs;
SEGMENT segments[MAX_TP_SEGMENTS];


int nspecial_cmds;
SPECIAL_CMD special_cmds[MAX_SPECIAL_CMDS];
int ispecial_cmd_downloaded;


double CalcChangeInDirection(int i);
double CalcLengthAlongCircle(double x0, double y0, 
					  double x1, double y1, 
					  double xc, double yc, BOOL DirIsCCW, 
					  double *radius, double *theta0, double *dtheta);

void CalcFinalDirectionOfSegment(SEGMENT *p,double &dx, double &dy, double &dz);
void CalcBegDirectionOfSegment(SEGMENT *p,double &dx, double &dy, double &dz);
void AdjustSegVelocity(int i);
int CombineSegments(double MaxLength);



// used to combine colinear segments

#define MAX_COMBINE 100
SEGMENT CombinedList[MAX_COMBINE];
int nCombined; 
double CollinearTolerance;



// initialize for new list of segments
// define endpoint to have zero vel

void tp_init(double tol)
{
	ispecial_cmd_downloaded=nsegs=nspecial_cmds=nCombined=0;
	CollinearTolerance=tol;
}


// insert new segment at the end.  Re-evaluate backwards
// through list to see if velocities could be increased

int tp_insert_linear_seg(double x0, double y0, double z0, double a0, 
						 double x1, double y1, double z1, double a1, 
						 double MaxVel, double MaxCombineLength)
{
	if (nsegs >= MAX_TP_SEGMENTS-1) return 1; // too many segments?
	
	double dx=x1-x0;
	double dy=y1-y0;
	double dz=z1-z0;
	double da=a1-a0;

	// add into the list

	segments[nsegs].type = SEG_LINEAR;
	
	segments[nsegs].x0 = x0;
	segments[nsegs].y0 = y0;
	segments[nsegs].z0 = z0;
	segments[nsegs].a0 = a0;
	
	segments[nsegs].x1 = x1;
	segments[nsegs].y1 = y1;
	segments[nsegs].z1 = z1;
	segments[nsegs].a1 = a1;

	segments[nsegs].dx=sqrt(dx*dx+dy*dy+dz*dz+da*da);
	segments[nsegs].OrigVel = MaxVel;
	segments[nsegs].vel=0.0f;
	segments[nsegs].ChangeInDirection = CalcChangeInDirection(nsegs);
	segments[nsegs].Done=FALSE;

	if (CombineSegments(MaxCombineLength))
	{
		// don't maximize the last segment
		// because it might change
		AdjustSegVelocity(nsegs-1);
		MaximizeSegments();
		nsegs++;
	}
	return 0;
}



void SetSegmentVelAccels(int i, double Vel, double Accel, double Decel)
{
	segments[i].MaxVel   = Vel;
	segments[i].MaxAccel = Accel;
	segments[i].MaxDecel = Decel;
}

void GetSegmentDirection(int i, double *dx, double *dy, double *dz, double *da)
{
	*dx = segments[i].x1 - segments[i].x0;
	*dy = segments[i].y1 - segments[i].y0;
	*dz = segments[i].z1 - segments[i].z0;
	*da = segments[i].a1 - segments[i].a0;
}



//
// area=(1/2) x Base x Height. Where the height is an altitude drawn from the base to the opposite angle. 
// This formula makes for a relatively easy calculation of the area of a triangle but it is rather difficult 
// to naturally find a triangle that is given in terms of at least one side (the base) and a height. 
// We typically can determine or are given the sides of a triangle when a triangle is present. 
// formula does exist that can calculate the area of a triangle when all three sides are known.
// This formula is attributed to Heron of Alexandria but can be traced back to Archimedes.
//
// This formula is represented by
// Area=SQRT(s(s-a)(s-b)(s-c)),
// where s=(a+b+c)/2 or perimeter/2. 
//
// start points of the 3 segments are used as the 3 points
//

bool CheckCollinear(SEGMENT *s0, SEGMENT *s1, SEGMENT *s2, double tol)
{
	double dx = s1->x0 - s0->x0;
	double dy = s1->y0 - s0->y0;
	double dz = s1->z0 - s0->z0;
	double da = s1->a0 - s0->a0;

	double a = sqrt(dx*dx+dy*dy+dz*dz*da*da);

	dx = s2->x0 - s1->x0;
	dy = s2->y0 - s1->y0;
	dz = s2->z0 - s1->z0;
	da = s2->a0 - s1->a0;

	double b = sqrt(dx*dx+dy*dy+dz*dz*da*da);

	dx = s2->x0 - s0->x0;
	dy = s2->y0 - s0->y0;
	dz = s2->z0 - s0->z0;
	da = s2->a0 - s0->a0;

	double c = sqrt(dx*dx+dy*dy+dz*dz*da*da);

	// if dist from beg to end is tiny, then 
	//    if both other sides tiny treat as collinear

	if (c < tol) return (a < tol && b < tol);


	double s = s=(a+b+c)/2;
	double h = 2.0 * sqrt(s*(s-a)*(s-b)*(s-c))/c;

	if (h > tol) return false;

	// if the height is small, there is a possibility
	// that the "middle" point is actually outside
	// of the begining and end and should be treated as
	// non-collinear.  We test for that by testing if the sum of the two 
	// "short" sides are longer than the base.

	if (a+b-c > tol) return false;

	return true;
}



// attempt to combine newly added segment if within tolerance

// return 0 if it was OK to combine

#define SPEED_TOL 0.01
#define ACCEL_TOL 0.04

int CombineSegments(double MaxLength)
{

	if (nCombined >= MAX_COMBINE || nsegs < 1 || segments[nsegs].type != SEG_LINEAR || segments[nsegs-1].type != SEG_LINEAR || 
		segments[nsegs].Done || segments[nsegs-1].Done || nCombined >= MAX_COMBINE)
	{
		// that's enough, clear the list and don't combine
		nCombined = 0;
		return 1;
	}

	// if already quite long then don't combine
	
	if (segments[nsegs-1].dx > MaxLength) 
	{
		nCombined = 0;
		return 1;
	}

	// max speeds must be very similar

	if (fabs(segments[nsegs].OrigVel - segments[nsegs-1].OrigVel) > SPEED_TOL * segments[nsegs-1].OrigVel) 
	{
		nCombined = 0;
		return 1;
	}

/*	if (fabs(segments[nsegs].MaxAccel - segments[nsegs-1].MaxAccel)/segments[nsegs-1].MaxAccel > ACCEL_TOL) 
	{
		nCombined = 0;
		return 1;
	}

	if (fabs(segments[nsegs].MaxAccel - segments[nsegs-1].MaxAccel)/segments[nsegs-1].MaxAccel > ACCEL_TOL) 
	{
		nCombined = 0;
		return 1;
	}
*/
	SEGMENT end;

	end.x0 =  segments[nsegs].x1;
	end.y0 =  segments[nsegs].y1;
	end.z0 =  segments[nsegs].z1;
	end.a0 =  segments[nsegs].a1;

	if (!CheckCollinear(&segments[nsegs-1],&segments[nsegs],&end,CollinearTolerance))
	{
		nCombined = 0;
		return 1;
	}

	// yes it is collinear

	// check all previous points also

	for (int i=0; i<nCombined; i++)
	{
		if (!CheckCollinear(&segments[nsegs-1],&CombinedList[i],&end,CollinearTolerance))
		{
			// not collinear with other points
			// clear the list and return

			nCombined=0;
			return 1;
		}
	}

	// yes, ok to combine

	// save the point that was eliminated

	CombinedList[nCombined].x0 = segments[nsegs].x0;
	CombinedList[nCombined].y0 = segments[nsegs].y0;
	CombinedList[nCombined].z0 = segments[nsegs].z0;
	CombinedList[nCombined].a0 = segments[nsegs].a0;
	nCombined++;

	// alter previous to include this one
	
	segments[nsegs-1].x1 = segments[nsegs].x1;
	segments[nsegs-1].y1 = segments[nsegs].y1;
	segments[nsegs-1].z1 = segments[nsegs].z1;
	segments[nsegs-1].a1 = segments[nsegs].a1;
	
	double dx=segments[nsegs-1].x1 - segments[nsegs-1].x0;
	double dy=segments[nsegs-1].y1 - segments[nsegs-1].y0;
	double dz=segments[nsegs-1].z1 - segments[nsegs-1].z0;
	double da=segments[nsegs-1].a1 - segments[nsegs-1].a0;

	// note keep velocity since it might already be non-zero
	// and the previous segment might be maxed out (Done)
	// and transitioning into it
	segments[nsegs-1].dx=sqrt(dx*dx+dy*dy+dz*dz+da*da);
	segments[nsegs-1].ChangeInDirection = CalcChangeInDirection(nsegs-1);
	segments[nsegs-1].Done=FALSE;

	return 0;
}



// Reduce the allowed velocity based on the change of direction
// between this segment and the previous.  The idea is to find the
// max const speed that we could transverse a circle where the
// radial acceleration would equal our allowed acceleration
//
// This helps with the situation where the break angles are
// very small (smooth path) but the curvature is so high as
// to violate our allowed acceleration
//
// For now, we assume that our allowed radial acceleration
// is the same as our tangential acceleration
//
// Assume our const speed = s
//
// After "corner" our radial velocity will s * sin(angle)
// the time per this segment will be d/s
//
// Thus the accel would be A = (s * sin(T))/(d/s)
//                               2
//             or          A =  s  sin(T) / d
//
//     and,    s = sqrt(A * d / sin(T))
 
   

void AdjustSegVelocity(int i)
{
	if (i<1) return;  // first doesn't have a change in direction

	double A = segments[i].MaxAccel;

	double sinT = sin(segments[i].ChangeInDirection);  // seg contains angle (radians) from previous

	if (sinT == 0.0) return;

	double d = segments[i].dx; // length of this vector

	double s = sqrt(A * d / sinT);

	if (s < segments[i].OrigVel) segments[i].MaxVel = s;
}


void CalcFinalDirectionOfSegment(SEGMENT *p,double &dx, double &dy, double &dz)
{
	if (p->type == SEG_LINEAR)
	{
		// simple linear case

		dx = p->x1 - p->x0;
		dy = p->y1 - p->y0;
		dz = p->z1 - p->z0;
	}
	else
	{
		// Arc case

		// calc direction (in xy plane) from
		// center of rotation to terminal point

		double dxc = p->x1 - p->xc;
		double dyc = p->y1 - p->yc;

		// then turn left 90 degrees if CCW, right if CW
		// (note dxy later will be neg if CW)

		dx = -dyc;
		dy =  dxc;

		// scale this vector to be length of total xy motion
		// right now it's length is the radius 

		double radius, theta0, dtheta;

		double dxy = CalcLengthAlongCircle(p->x0, p->y0, p->x1, p->y1, 
					  p->xc, p->yc, p->DirIsCCW, &radius, &theta0, &dtheta);

		dx *= dxy/radius;
		dy *= dxy/radius;
		dz = p->z1 - p->z0;
	}
}


void CalcBegDirectionOfSegment(SEGMENT *p,double &dx, double &dy, double &dz)
{
	if (p->type == SEG_LINEAR)
	{
		// simple linear case

		dx = p->x1 - p->x0;
		dy = p->y1 - p->y0;
		dz = p->z1 - p->z0;
	}
	else
	{
		// Arc case

		// calc direction (in xy plane) from
		// center of rotation to beginning point

		double dxc = p->x0 - p->xc;
		double dyc = p->y0 - p->yc;

		// then turn left 90 degrees if CCW, right if CW
		// (note dxy later will be neg if CW)

		dx = -dyc;
		dy =  dxc;

		// scale this vector to be length of total xy motion
		// right now it's length is the radius 

		double radius, theta0, dtheta;

		double dxy = CalcLengthAlongCircle(p->x0, p->y0, p->x1, p->y1, 
					  p->xc, p->yc, p->DirIsCCW, &radius, &theta0, &dtheta);

		dx *= dxy/radius;
		dy *= dxy/radius;
		dz = p->z1 - p->z0;
	}
}




double CalcChangeInDirection(int i)
{
	if (i==0) return 0.0;  // return if first segment

	double adx,ady,adz,bdx,bdy,bdz;

	SEGMENT *a=&segments[i-1];
	SEGMENT *b=&segments[i];

	if (a->dx == 0.0 || b->dx == 0.0) return 0.0;

	CalcFinalDirectionOfSegment(a,adx,ady,adz);
	CalcBegDirectionOfSegment(b,bdx,bdy,bdz);

	double dot = (adx) * (bdx) +
		         (ady) * (bdy) +
		         (adz) * (bdz);

	dot /= a->dx * b->dx;

	if (dot >  1.0) return 0.0;
	if (dot < -1.0) return PI;

	return acos(dot);
}


#define THETA_SIGMA 1e-13

double CalcLengthAlongHelix(double x0, double y0, double z0, 
					  double x1, double y1, double z1, 
					  double xc, double yc, BOOL DirIsCCW, 
					  double *radius, double *theta0, double *dtheta)
{
	double dz,dxy;

	dxy = CalcLengthAlongCircle(x0,y0,x1,y1,xc,yc,DirIsCCW,radius,theta0,dtheta);

	dz=z1-z0;

	return sqrt(dxy*dxy + dz*dz); // total length
}


double CalcLengthAlongCircle(double x0, double y0, 
					  double x1, double y1, 
					  double xc, double yc, BOOL DirIsCCW, 
					  double *radius, double *theta0, double *dtheta)
{	
	double t0,t1,dt,dx,dy,r;

	dx=x1-xc;
	dy=y1-yc;

	r = sqrt(dx*dx + dy*dy); 

	t0 = atan2(y0-yc,x0-xc);
	t1 = atan2(y1-yc,x1-xc);  
	                    
	dt =  t1 - t0;  
	
	if (fabs(dt) < THETA_SIGMA) dt=0;  // avoid confusing arc to same place with full circle

	if (DirIsCCW)
	{
		if (dt < 0.0f) dt+=TWO_PI; // CCW delta should be +  
	}
	else
	{
		if (dt > 0.0f) dt-=TWO_PI;  // CW delta should be -
	}
	
	*radius = r;  // return these to caller
	*theta0 = t0;
	*dtheta = dt;

	return  dt * r;  // path length along circle                                   
}



// insert new segment at the end.  Re-evaluate backwards
// through list to see if velocities could be increased

int tp_insert_arc_seg(double x0, double y0, double z0, 
					  double x1, double y1, double z1, 
					  double xc, double yc, BOOL DirIsCCW,
					  double MaxVel, double MaxAccel, double MaxDecel)
{
	double radius, theta0, dtheta;
	
	if (nsegs >= MAX_TP_SEGMENTS-1) return 1; // too many segments?


	// add into the list

	segments[nsegs].type = SEG_ARC;
	
	segments[nsegs].x0 = x0;
	segments[nsegs].y0 = y0;
	segments[nsegs].z0 = z0;
	
	segments[nsegs].x1 = x1;
	segments[nsegs].y1 = y1;
	segments[nsegs].z1 = z1;

	segments[nsegs].xc = xc;
	segments[nsegs].yc = yc;
	
	segments[nsegs].DirIsCCW = DirIsCCW;

	segments[nsegs].dx=CalcLengthAlongHelix(x0,y0,z0,x1,y1,z1,xc,yc,DirIsCCW,&radius,&theta0,&dtheta);  // total length
	segments[nsegs].MaxVel = segments[nsegs].OrigVel = MaxVel;
	segments[nsegs].MaxAccel=MaxAccel;
	segments[nsegs].MaxDecel=MaxDecel;
	segments[nsegs].vel=0.0f;

	segments[nsegs].ChangeInDirection = CalcChangeInDirection(nsegs);

	segments[nsegs++].Done=FALSE;

	MaximizeSegments();

	return 0;
}



// move backwards through the list and update velocities based on pairs


void MaximizeSegments()
{
	bool SomethingChanged;
	int i;
	double V0_b,VM_b,A_b,X_b,t_b,d_b;
	double V0_e,VM_e,A_e,X_e,t_e,d_e;
	double MaxEndVel0,MaxBegVel1;

	do 
	{
		SomethingChanged=false;

		for (i=nsegs-1; i>0 && !segments[i-1].Done; i--)
		{
			// consider two segments
			//
			//        (i-1)         (i)
			//        seg0         seg1
			//   X------------X------------X
			//  beg0         end0
			//               beg1         end1
			//
			//
			// assume velocities at beg0 and end1 are known
			//
			// calc max possible vel that could be achieved
			// by accelerating from beg0 to end0 (MaxEndVel0)
			//
			// calc max possible vel that we could have at
			// beg1 and still decel to end1 velocity (MaxBegVel1)
			//
			// we must use the lesser of the two.  once this
			// is known, calculate the motion through both
			// segments to achieve this



			// Calc MaxEndVel0
			// if there is no previous segment (i=0), set to zero 
			// calc how far we would travel if we maintained
			// MaxAccel until we achieved MaxVel

			if (i==0 || fabs(segments[i].ChangeInDirection) > BreakAngle)
			{
				MaxEndVel0 = 0.0f;
			}
			else
			{
				V0_b = segments[i-1].vel;
				VM_b = segments[i-1].MaxVel;
				A_b  = segments[i-1].MaxAccel;
				X_b  = segments[i-1].dx;

				// time to achieve max vel
				t_b = (VM_b - V0_b)/A_b;

				// dist to achieve max vel
				d_b = (V0_b + 0.5f * A_b * t_b) * t_b;

				if (X_b > d_b)
				{
					// we need const vel phase

					MaxEndVel0 = VM_b;
				}
				else
				{
					// must solve quadratic to determine time at end

					t_b = (-V0_b + sqrt(V0_b*V0_b + 2.0f*A_b*X_b))/A_b;

					MaxEndVel0 = V0_b + t_b * A_b;
				}
			}

			// Calc MaxBegVel1
			// (solve prob as if time is running backward)
			// calc how far we would travel if we maintained
			// MaxDeccel until we achieved MaxVel

			// if there is a seg following, get its init vel
			// otherwise we must stop

			if (i<nsegs-1)
				V0_e = segments[i+1].vel;
			else
				V0_e = 0.0f;

			VM_e = segments[i].MaxVel;
			A_e  = segments[i].MaxDecel;
			X_e  = segments[i].dx;

			// time to achieve max vel
			t_e = (VM_e - V0_e)/A_e;

			// dist to achieve max vel
			d_e = (V0_e + 0.5f * A_e * t_e) * t_e;

			if (X_e > d_e)
			{
				// we need const vel phase

				MaxBegVel1 = VM_e;
			}
			else
			{
				// must solve quadratic to determine time at end

				t_e = (-V0_e + sqrt(V0_e*V0_e + 2.0f*A_e*X_e))/A_e;

				MaxBegVel1 = V0_e + t_e * A_e;
			}

			// check which is limiting

			if (MaxEndVel0 <= MaxBegVel1)
			{
				// first segment is limiting velocity (MaxEndVel0 will be used).

				// check if the i-1th segment can be determined to be "Done" :
				// (ending velocity won't ever change)
				//
				// if it ends at MaxVel for the segment or
				// if the previous is Done and MaxAccel occurs
				// throughout the segment
				// 
				// The fact that the first seg is limiting velocity
				// is sufficient that it is maxed out.
				
				if (i-1 == 0 || segments[i-2].Done)
				{
					if (!segments[i-1].Done)
					{
						segments[i-1].Done = true;
						SomethingChanged = true;
					}
				}
				// if the value increased another pass might be necessary
				if (MaxEndVel0 > segments[i].vel) 
				{
					SomethingChanged = true;
					segments[i].vel = MaxEndVel0;
				}
			}
			else
			{
				// second segment is limiting velocity.
				
				// if the value increased another pass might be necessary
				if (MaxBegVel1 > segments[i].vel)
				{
					segments[i].vel = MaxBegVel1;
					SomethingChanged = true;
				}

				// Any segment that has a Done segment or segment already beginning at
				// max velocity to the right, and is limited by the right side is
				// Done.

				// Note: don't necessarily flag the segment that can begin
				// at max velocity as done because it might not be determined
				// whether it should end at max velocity or some slower speed

				if (segments[i].Done || segments[i].vel == segments[i].MaxVel)
				{
					if (!segments[i-1].Done)
					{
						segments[i-1].Done= true;
						SomethingChanged = true;
					}
				}
			}
		}
	} while (SomethingChanged);
}




// calculate the trip states (three 2nd order polynomials)
// for a segment given the initial and ending velocities

int tp_calc_seg_trip_states(int i)
{
	double V0,V1,VM,X,A,D,ta,td,da,dd,tc;

	V0 = segments[i].vel;

	if (i<nsegs-1)
		V1 = segments[i+1].vel;
	else
		V1 = 0.0f;

	VM = segments[i].MaxVel;
	A  = segments[i].MaxAccel;
	D  = segments[i].MaxDecel;
	X  = segments[i].dx;

	if (VM==0 || A==0 || D==0)
	{
		AfxMessageBox("Trajectory Planner has Invalid Velocity or Acceleration",MB_ICONSTOP|MB_OK);
		return 1;
	}


	// time to achieve max vel
	ta = (VM - V0)/A;

	// dist to achieve max vel
	da = (V0 + 0.5f * A * ta) * ta;

	// time to decel from max vel
	td = (VM - V1)/D;

	// dist to decel from max vel
	dd = (V1 + 0.5f * D * td) * td;

	if (X > da + dd)
	{
		// we need const vel phase

		tc = (X-da-dd)/VM;
	}
	else
	{
		// must solve to determine time at highest vel

		VM = sqrt((A*V1*V1+D*V0*V0+2.0f*A*D*X)/(A+D));

		ta = (VM-V0)/A;
		td = (VM-V1)/D;
		tc=0.0f;
	}

	segments[i].C[0].a = 0.5f*A;     // phase 0 - Accelerate
	segments[i].C[0].b = V0;
	segments[i].C[0].c = 0;          // parametric value relative to seg beginning
	segments[i].C[0].t = ta;
	
	segments[i].C[1].a = 0.0f;       // phase 1 - Const vel
	segments[i].C[1].b = VM;
	segments[i].C[1].c = 0.5f*A*ta*ta + V0*ta; 
	segments[i].C[1].t = tc;
	
	segments[i].C[2].a = -0.5f*D;    // phase 2 - Decel
	segments[i].C[2].b = VM;
	segments[i].C[2].c = segments[i].C[1].c + VM*tc; 
	segments[i].C[2].t = td;

	return 0;
}