// GCodeInterpreter.cpp : Defines the entry point for the DLL application.
//

#include "stdafx.h"
#include "rs274ngc_return.h"

CCoordMotion *CM;
CGCodeInterpreter *GC;

CGCodeInterpreter::CGCodeInterpreter()
{ 
	CM=&CoordMotion;  // set global variable for the low level Interpreter
	GC=this;

	ToolFile[0] = 0;
	SetupFile[0] = 0;

	m_CurrentLine=0;

	p_setup = &_setup;

	for (int i=0; i<MAX_MCODE_ACTIONS; i++)
	{
		McodeActions[i].Action=0;
		
		for (int k=0; k<MAX_MCODE_DOUBLE_PARAMS; k++)
			McodeActions[i].dParams[k]=0;
		
		McodeActions[i].String[0]=0;
	}

	p_setup->length_units=CANON_UNITS_INCHES;
	p_setup->origin_index=1;
	p_setup->tool_table_index=1;

	m_InitializeOnExecute = true; // make sure we initialize at least once
}

CGCodeInterpreter::~CGCodeInterpreter()
{
}


int CGCodeInterpreter::Interpret(int board,
		      const char *fname,
			  int start, int end,
			  int restart,
              G_STATUS_CALLBACK *StatusFn,
			  G_COMPLETE_CALLBACK *CompleteFn)
{
	CoordMotion.m_board=board;
	CoordMotion.ClearAbort();
	CoordMotion.m_AxisDisabled=false;

	m_CurrentLine=m_start=start;
	m_end=end;
	m_restart=restart;
	m_CompleteFn=CompleteFn;
	m_StatusFn=StatusFn;
	m_InFile=fname;
	m_Halt=false;

	LaunchExecution();	
	
	return 0;
}

// returns a pointer to the GCode Parameters

MOTION_PARAMS *CGCodeInterpreter::GetMotionParams()
{
	return &CoordMotion.Kinematics->m_MotionParams;
}



DWORD DoExecuteShell(LPDWORD lpdwParam)
{
	CGCodeInterpreter *p=(CGCodeInterpreter*)lpdwParam;

	p->m_exitcode = p->DoExecute();

	p->DoExecuteComplete();

	return 0;
}


void CGCodeInterpreter::Halt()
{
	m_Halt=true;
}

bool CGCodeInterpreter::GetHalt()
{
	return m_Halt;
}

void CGCodeInterpreter::Abort()
{
	CoordMotion.SetAbort();
}

bool CGCodeInterpreter::GetAbort()
{
	return CoordMotion.GetAbort();
}

void CGCodeInterpreter::SetToolFile(char *f)
{
	strncpy(f,ToolFile,MAX_PATH);
}

void CGCodeInterpreter::SetSetupFile(char *f)
{
	strncpy(f,SetupFile,MAX_PATH);
}


int CGCodeInterpreter::rs274ErrorExit(int status)
{
	CString ErrDescr;

	rs274ngc_close();

	if (CoordMotion.GetAbort())
	{
		if (CoordMotion.m_AxisDisabled)
		{
			ErrDescr="Axis Disabled - GCode Aborted";
			status=1000;
		}
		else
		{
			ErrDescr="GCode Aborted";
			status=1001;
		}
	}
	else
	{
		rs274ngc_error_text(status,ErrDescr.GetBuffer(200),200);
		ErrDescr.ReleaseBuffer();
	}

	ErrorOutput+=ErrDescr;

	return status;
}


int CGCodeInterpreter::ReadSetupFile()
{
	int status = RS274NGC_OK;
	
	if (SetupFile[0]!=0)
	{
		status = read_setup_file(SetupFile, &_setup);
	}
	
	return status;
}

int CGCodeInterpreter::DoExecute()
{
	int status;
	char trash[INTERP_TEXT_SIZE];
	char * read_ok;
	int program_status;
	CString OutputCRLF;

	ErrorOutput="";

	program_status = RS274NGC_OK;

	// initialize everything if we are starting at the beginning

	if (m_restart || m_InitializeOnExecute)
	{
		Output="";
		line_number=0;

		tp_init(CoordMotion.Kinematics->m_MotionParams.CollinearTol);					 // initialize the trajectory planner
		CoordMotion.DownloadInit();  // intialize download/look ahead variables  

		
		status = rs274ngc_init();
		if (status != RS274NGC_OK)	return rs274ErrorExit(status);

		
		if (ToolFile[0]!=0)
		{
			status = read_tool_file(ToolFile, &_setup);
			if (status != RS274NGC_OK)	return rs274ErrorExit(status);
		}

		if (SetupFile[0]!=0)
		{
			status = read_setup_file(SetupFile, &_setup);
			if (status != RS274NGC_OK)	return rs274ErrorExit(status);
		}
		m_InitializeOnExecute = false;
	}

	if (!CoordMotion.m_Simulate && CoordMotion.ReadAndSyncCurPositions())
	{
		if (CoordMotion.m_AxisDisabled)
			ErrorOutput="Unable to read defined coordinate system axis positions - Axis Disabled";
		else
			ErrorOutput="Unable to read defined coordinate system axis positions";

		return 1;
	}


	status = rs274ngc_open(m_InFile.GetBuffer(0)); 
	if (status != RS274NGC_OK)	return rs274ErrorExit(status);

	if (_setup.percent_flag == ON)
		m_GCodeReads = 1;
	else
		m_GCodeReads = 0;
			
	for( ; m_GCodeReads<m_CurrentLine ; m_GCodeReads++)
	{
		read_ok = fgets(trash, INTERP_TEXT_SIZE,_setup.file_pointer);
		if (!read_ok) 
		{
			ErrorOutput="Error while reading GCode file";
			return NCE_A_FILE_IS_ALREADY_OPEN;
		}
	}

	
	for(  ; ; m_GCodeReads++)
	{
		// record what line we are doing in the GCode
		_setup.current_line = m_CurrentLine = m_GCodeReads;
		
		// give output to caller			
		Output.Replace("\n","\r\n");

		m_StatusFn(m_CurrentLine,Output);
		
		Output="";  // clear it


		if (((m_end!=-1)&&(m_CurrentLine>m_end)) || m_Halt || CoordMotion.GetAbort())
		{
			rs274ngc_close();
			return 0;
		}

		status = rs274ngc_read();
		if (status == RS274NGC_ENDFILE)
		{
			rs274ngc_close();
			return RS274NGC_ENDFILE;
		}

		if (status != RS274NGC_OK)	return rs274ErrorExit(status);

		status = rs274ngc_execute(NULL);

		if (m_CurrentLine != _setup.current_line)
		{
			// line number changed

			if (m_end!=-1)
				m_end += _setup.current_line - m_CurrentLine;
		
			m_GCodeReads = m_CurrentLine = _setup.current_line;  // get back line no (in case we called a subroutine)
		}

		if (CoordMotion.GetAbort()) return rs274ErrorExit(status);
		
		if (status == RS274NGC_EXIT)
		{
			rs274ngc_close();
			m_CurrentLine = 0;  // reset back to the beginning
			return program_status;
		}
	
		if (status != RS274NGC_OK)	return rs274ErrorExit(status);
	}

	return 0;
}


int CGCodeInterpreter::DoExecuteComplete()
{
	CoordMotion.ExecutionStop();
	m_CompleteFn(m_exitcode,m_CurrentLine,ErrorOutput);

	return 0;
}

int CGCodeInterpreter::LaunchExecution()
{
	DWORD ID;
	

	// Now start a worker thread that performs the G Code
	
	HANDLE Thread=CreateThread( 
		NULL,                        /* no security attributes        */ 
		0,                           /* use default stack size        */ 
		(LPTHREAD_START_ROUTINE) ::DoExecuteShell, /* thread function       */ 
		this,							 /* argument to thread function   */ 
		0,                           /* use default creation flags    */ 
		&ID);   

	return 0;
}

	

// invoke an MCode action to Set A Bit, Dac, execute a program, etc...

int CGCodeInterpreter::InvokeAction(int i, BOOL FlushBeforeUnbufferedOperation)
{
	MCODE_ACTION *p=&McodeActions[i];
	CString s;
	double value;
	int ivalue;

	switch (p->Action)
	{
	case M_Action_None:
		break;

	case M_Action_Setbit:
		s.Format("SetStateBitBuf%d=%d",(int)p->dParams[0],(int)p->dParams[1]);
		if (CoordMotion.DoKMotionBufCmd(s)) return 1;
		break;

	case M_Action_SetTwoBits:
		s.Format("SetStateBitBuf%d=%d",(int)p->dParams[0],(int)p->dParams[1]);
		if (CoordMotion.DoKMotionBufCmd(s)) return 1;
		s.Format("SetStateBitBuf%d=%d",(int)p->dParams[2],(int)p->dParams[3]);
		if (CoordMotion.DoKMotionBufCmd(s)) return 1;
		break;

	case M_Action_DAC:
		value = p_setup->speed * p->dParams[1] + p->dParams[2];   // scale and offset
		ivalue = (int)floor(value+0.5); 
		if (ivalue < (int)p->dParams[3]) ivalue = (int)p->dParams[3];
		if (ivalue > (int)p->dParams[4]) ivalue = (int)p->dParams[4];
		s.Format("DAC%d=%d",(int)p->dParams[0],(int)p->dParams[1]);
		if (CoordMotion.DoKMotionCmd(s)) return 1;
		break;

	case M_Action_Program:
		if (FlushBeforeUnbufferedOperation)
		{
			if (CoordMotion.FlushSegments()) {CoordMotion.SetAbort(); return 1;}  
			if (CoordMotion.WaitForSegmentsFinished(TRUE)) {CoordMotion.SetAbort(); return 1;}
		}

		// set a persist variable to either the Mcode value
		// or the related variable (normally speed)
		// if variable is -1 then don't set anything

		if (p->dParams[1] >= 0)
		{
			if (i==6)  // tool change
			{
				s.Format("SetPersistHex %d %x",(int)p->dParams[1], p_setup->tool_table_index);
			}
			else if (i==10)  // set speed
			{
				float fspeed = (float)p_setup->speed;
				s.Format("SetPersistHex %d %x",(int)p->dParams[1], *(int *)&fspeed);
			}
			else
			{
				s.Format("SetPersistHex %d %x",(int)p->dParams[1],i);
			}

			if (CoordMotion.KMotionDLL.WriteLine(CoordMotion.m_board,s)) {CoordMotion.SetAbort(); return 1;}
		}

		// If a C File is specified then Compile and load it

		if (p->String[0])
		{
			CString Err;

			if (CoordMotion.KMotionDLL.CompileAndLoadCoff(CoordMotion.m_board, p->String, (int)p->dParams[0], Err.GetBuffer(500), 499))
			{
				Err.ReleaseBuffer();
				AfxMessageBox("Error Compiling and Loading KMotion Program\r\r" + ((CString)p->String) + "\r\r" + Err);
				return 1;
			}
		}

		// Now execute the thread!

		s.Format("Execute%d",(int)p->dParams[0]);
		if (CoordMotion.KMotionDLL.WriteLine(CoordMotion.m_board,s)) {CoordMotion.SetAbort(); return 1;}

		break;
	}
	return 0;
}

double CGCodeInterpreter::ConvertAbsToUserUnitsX(double x)
{
	return InchesToUserUnits(x - p_setup->axis_offset_x  - p_setup->origin_offset_x);
}

double CGCodeInterpreter::ConvertAbsToUserUnitsY(double y)
{
	return InchesToUserUnits(y - p_setup->axis_offset_y  - p_setup->origin_offset_y);
}

double CGCodeInterpreter::ConvertAbsToUserUnitsZ(double z)
{
	return InchesToUserUnits(z - p_setup->axis_offset_z  - p_setup->origin_offset_z);
}

double CGCodeInterpreter::ConvertAbsToUserUnitsA(double a)
{
	return InchesToUserUnits(a - p_setup->AA_axis_offset  - p_setup->AA_origin_offset);
}

double CGCodeInterpreter::InchesToUserUnits(double inches)
{
	return p_setup->length_units==CANON_UNITS_INCHES ? inches : inches * 25.4;
}

double CGCodeInterpreter::UserUnitsToInches(double inches)
{
	return p_setup->length_units==CANON_UNITS_INCHES ? inches : inches / 25.4;
}



	
// Call with fixture number 1 through 9
//
// 1 = G54 = default 
// 2 = G55 
// 3 = G56 
// 4 = G57 
// 5 = G58 
// 6 = G59 
// 7 = G59.1 
// 8 = G59.2 
// 9 = G59.3 

int CGCodeInterpreter::ChangeFixtureNumber(int fixture)
{
	switch (fixture)
	{
	case 1:	return rs274ngc_execute("G54");
	case 2:	return rs274ngc_execute("G55");
	case 3:	return rs274ngc_execute("G56");
	case 4:	return rs274ngc_execute("G57");
	case 5:	return rs274ngc_execute("G58");
	case 6:	return rs274ngc_execute("G59");
	case 7:	return rs274ngc_execute("G59.1");
	case 8:	return rs274ngc_execute("G59.2");
	case 9:	return rs274ngc_execute("G59.3");
	}
	return 0;
}
