// CoordMotion.cpp: implementation of the CCoordMotion class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "CoordMotion.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CCoordMotion::CCoordMotion()
{
	m_board=0;

	current_x=current_y=current_z=current_a=0;

	m_FeedRateOverride=1.0;
	m_Simulate=false;
	m_DefineCS_valid=false;

	// Save for everybody what directory we are installed in

	int nParams;
	LPWSTR *CL= CommandLineToArgvW(GetCommandLineW(),&nParams);
	CString Path = CL[0];
	GlobalFree(CL);

	Path.Replace("\"","");  // remove quotes
	Path.TrimRight();
	Path.TrimLeft();

	int LastSlash=Path.ReverseFind('\\');
	Path=Path.Left(LastSlash);

	// Check if we are running from the debug directory
	// if we are, then strip it off

	if (Path.Right(6).CompareNoCase("\\debug") == 0)
	{
		Path = Path.Left(Path.GetLength()-6);
	}

	// Check if we are running from the release directory
	// if we are, then strip it off

	if (Path.Right(8).CompareNoCase("\\release") == 0)
	{
		Path = Path.Left(Path.GetLength()-8);
	}

	// Now set the root install directory

	if (Path.Right(8).CompareNoCase("\\KMotion") == 0)
	{
		strncpy(MainPathRoot,Path.Left(Path.GetLength()-8),MAX_PATH);
	}

	strncpy(MainPath,Path,MAX_PATH);

	m_StraightTraverseCallback=NULL;
	m_StraightFeedCallback=NULL;
	m_ArcFeedCallback=NULL;

	DownloadInit();

	current_x = current_y = current_z = current_a = 0.0;

	m_SegmentsStartedExecuting = m_Abort = false;


	// check for a special Kinematics File

	FILE *f = fopen((CString)MainPath + "\\Data\\Kinematics.txt","rt");

	if (f)
	{
		// one exists, assume it is the 3Rod

		Kinematics = new CKinematics3Rod;
		fclose(f);
	}
	else
	{
		Kinematics = new CKinematics;
	}


}

CCoordMotion::~CCoordMotion()
{
	if (Kinematics) delete Kinematics;
}

MOTION_PARAMS *CCoordMotion::GetMotionParams()
{
	return &Kinematics->m_MotionParams;
}

int CCoordMotion::StraightTraverse(double x, double y, double z, double a)
{
	if (m_StraightTraverseCallback) m_StraightTraverseCallback(x,y,z,a);

	if (m_Simulate)
	{
		current_x = x;
		current_y = y;
		current_z = z;
		current_a = a;
		return 0;  // exit if we are simulating
	}


	if (FlushSegments()) {SetAbort(); return 1;}  

	if (WaitForSegmentsFinished()) {SetAbort(); return 1;}

	double Acts[MAX_ACTUATORS];
	if (Kinematics->TransformCADtoActuators(x,y,z,a,Acts)) return 1;

	CString s;
	s.Format("MoveXYZA %f %f %f %f", Acts[0],Acts[1],Acts[2],Acts[3]); 

	current_x = x;
	current_y = y;
	current_z = z;
	current_a = a;

	if (KMotionDLL.WriteLine(m_board,s)) {SetAbort(); return 1;}

	// wait until complete	

	if (WaitForMoveXYZAFinished()) {SetAbort(); return 1;}

	return 0;
}



int CCoordMotion::ArcFeed(double DesiredFeedRate_in_per_sec, 
				double first_end, double second_end, 
		        double first_axis, double second_axis, int rotation,
				double axis_end_point)
{
	double radius,theta0,dtheta,MaxLength;
	double FeedRateToUse = DesiredFeedRate_in_per_sec * m_FeedRateOverride;
	double AccelToUse = Kinematics->m_MotionParams.MaxAccelX;


	if (Kinematics->m_MotionParams.MaxAccelY < AccelToUse) AccelToUse = Kinematics->m_MotionParams.MaxAccelY;

	if (Kinematics->m_MotionParams.MaxVelX < FeedRateToUse) FeedRateToUse = Kinematics->m_MotionParams.MaxVelX;
	if (Kinematics->m_MotionParams.MaxVelY < FeedRateToUse) FeedRateToUse = Kinematics->m_MotionParams.MaxVelY;


	double d=CalcLengthAlongHelix(current_x,current_y,current_z,
		                          first_end,second_end,axis_end_point,first_axis,second_axis,
								  rotation,&radius,&theta0,&dtheta);  // total length

	// if this move is expected to take more than 1/2th of the download time then break int0 2 parts (recursively)
	// (or if greater than the Kinematics length (line might actually be curved in actuator space)

	MaxLength = FeedRateToUse * Kinematics->m_MotionParams.TPLookahead/2.0;
	if (MaxLength < Kinematics->m_MotionParams.MaxLinearLength) MaxLength = Kinematics->m_MotionParams.MaxLinearLength;

	if (!m_Simulate && d > MaxLength)
	{
		double theta = theta0 + dtheta/2.0;

		int result = ArcFeed(DesiredFeedRate_in_per_sec, first_axis  + radius * cos(theta),
														 second_axis + radius * sin(theta),
														 first_axis, second_axis, rotation, 
														 (current_z+axis_end_point)/2.0);
		if (result) return result;
		
		return ArcFeed(DesiredFeedRate_in_per_sec, first_end, second_end, 
		        first_axis, second_axis, rotation, axis_end_point);
	}
	
	if (m_ArcFeedCallback) m_ArcFeedCallback(DesiredFeedRate_in_per_sec, 
				first_end, second_end, 
		        first_axis, second_axis, rotation,
				axis_end_point,
				current_x,current_y,current_z);

	if (m_Simulate)
	{
		current_x = first_end;
		current_y = second_end;
		current_z = axis_end_point;
		return 0;  // exit if we are simulating
	}
	
	
	
	BreakAngle = Kinematics->m_MotionParams.BreakAngle * PI/180.0;

	tp_insert_arc_seg(current_x, 
					  current_y, 
					  current_z, 
					  first_end, second_end, axis_end_point,
					  first_axis, second_axis, rotation,
					  FeedRateToUse, 
					  AccelToUse, 
					  AccelToUse);

	current_x = first_end;
	current_y = second_end;
	current_z = axis_end_point;

	if (CheckForOKToFlushSegments()) {SetAbort(); return 1;}
	if (DownloadDoneSegments()) {SetAbort(); return 1;}

	return 0;
}





int CCoordMotion::StraightFeed(double DesiredFeedRate_in_per_sec,
							   double x, double y, double z, double a)
{
	double FeedRateToUse = DesiredFeedRate_in_per_sec * m_FeedRateOverride;
	double AccelToUse,MaxLength;

	double dx = x - current_x;
	double dy = y - current_y;
	double dz = z - current_z;
	double da = a - current_a;

	double d = sqrt(dx*dx+dy*dy+dz*dz+da*da);

	if (d==0.0) return 0;  // ignore zero length moves

	// if this move is expected to take more than 1/2th of the download time then break int0 2 parts (recursively)
	// (or if greater than the Kinematics length (line might actually be curved in actuator space)

	MaxLength = FeedRateToUse * Kinematics->m_MotionParams.TPLookahead/2.0;
	if (MaxLength < Kinematics->m_MotionParams.MaxLinearLength) MaxLength = Kinematics->m_MotionParams.MaxLinearLength;

	if (!m_Simulate && d > MaxLength)
	{
		int result = StraightFeed(DesiredFeedRate_in_per_sec, current_x+dx/2.0, current_y+dy/2.0, current_z+dz/2.0, current_a+da/2.0);
		if (result) return result;
		
		return StraightFeed(DesiredFeedRate_in_per_sec, x, y, z, a);
	}


	if (m_StraightFeedCallback) m_StraightFeedCallback(DesiredFeedRate_in_per_sec,x,y,z,a);

	if (m_Simulate) 
	{
		current_x  = x;
		current_y  = y;
		current_z  = z;
		current_a  = a;
		return 0;  // exit if we are simulating
	}

	
	BreakAngle = Kinematics->m_MotionParams.BreakAngle * PI/180.0;

	// add in the segment to the planner
	tp_insert_linear_seg(current_x, 
						 current_y, 
						 current_z, 
						 current_a, 
						 x, y, z, a,
						 FeedRateToUse, 
						 MaxLength);

	// limit speeds based on proportion in that direction
	// since the segment might have been combined
	// compute the max velocities and accelerations
	// for the possibly new direction

	double tdx,tdy,tdz,tda,rate;
	
	GetSegmentDirection(nsegs-1, &tdx, &tdy, &tdz, &tda);
	
	if (Kinematics->MaxRateInDirection(tdx,tdy,tdz,tda,&rate)) return 1;
	
	if (rate < FeedRateToUse) FeedRateToUse = rate;

	// limit accel based on proportion in that direction
	if (Kinematics->MaxAccelInDirection(dx,dy,dz,da,&AccelToUse)) return 1;
	SetSegmentVelAccels(nsegs-1, FeedRateToUse, AccelToUse, AccelToUse);

	current_x  = x;
	current_y  = y;
	current_z  = z;
	current_a  = a;

	if (CheckForOKToFlushSegments()) {SetAbort(); return 1;}
	if (DownloadDoneSegments()) {SetAbort(); return 1;}

	return 0;
}



int CCoordMotion::Dwell(double seconds)
{
	if (m_Simulate) return 0;  // exit if we are simulating
	
	if (FlushSegments()) {SetAbort(); return 1;}  

	if (WaitForSegmentsFinished()) {SetAbort(); return 1;}

	if(KMotionDLL.WriteLine(m_board,"OpenBuf")) {SetAbort(); return 1;}
	m_SegmentsStartedExecuting = false;
	m_TotalDownloadedTime = m_TimeAlreadyExecuted = 0.0;
	m_nsegs_downloaded=0;

	CString s;

	// insert a dummy linear move that has no length but takes
	// the dwell time to move
	
	double Acts[MAX_ACTUATORS];
	if (Kinematics->TransformCADtoActuators(current_x,current_y,current_z,current_a,Acts)) return 1;

	s.Format("Linear %f %f %f %f %f %f %f %f %f %f %f %f %f",
		Acts[0],
		Acts[1],
		Acts[2],
		Acts[3],
		Acts[0],
		Acts[1],
		Acts[2],
		Acts[3],
		0.0f,
		0.0f,
		0.0f,
		0.0f,
		seconds);

	if (KMotionDLL.WriteLine(m_board,s)) {SetAbort(); return 1;}
		
	if (KMotionDLL.WriteLine(m_board,"ExecBuf")){SetAbort(); return 1;}
	m_SegmentsStartedExecuting = true;

	return 0;
}



// issue a pass through KMotion Command

int CCoordMotion::DoKMotionCmd(const char *s)
{
	if (m_Simulate) return 0;  // exit if we are simulating

	if (FlushSegments()) {SetAbort(); return 1;}  

	if (WaitForSegmentsFinished()) {SetAbort(); return 1;}

	if (KMotionDLL.WriteLine(m_board,s)) {SetAbort(); return 1;}

	return 0;
}


// issue a buffered KMotion Command

int CCoordMotion::DoKMotionBufCmd(const char *s)
{
	strncpy(special_cmds[nspecial_cmds].cmd,s,MAX_LINE);
	special_cmds[nspecial_cmds++].seg_before=nsegs;
	return 0;
}


// Check if we came to a stop, if so then it is OK to
// execute the previous segments


int CCoordMotion::CheckForOKToFlushSegments()
{
	// note last one added might be very short (and noisy) so don't
	// consider it's angle until we get another and determine that
	// more can't be combined
	if (nsegs>1 && fabs(segments[nsegs-2].ChangeInDirection) > BreakAngle)
	{
		SEGMENT SaveSeg = segments[nsegs-1];  // save the first seg of the new sequence
		nsegs--;                              // remove it from the list
		if (FlushSegments()) {SetAbort(); return 1;}        // execute the list
		segments[0]=SaveSeg;				  // put the new one at the beginning
		nsegs=1;
	}
	return 0;
}


int CCoordMotion::WaitForSegmentsFinished(BOOL NoErrorOnDisable)
{
	CString response;

	int count=0;

	do
	{
		if (count++) Sleep(10);
		
		if (KMotionDLL.WriteLineReadLine(m_board,"CheckDoneBuf",response.GetBufferSetLength(MAX_LINE))) {SetAbort(); return 1;}
		response.ReleaseBuffer();

		if (response == "-1")
		{
			if (NoErrorOnDisable)
			{
				return 0;
			}
			else
			{
				m_AxisDisabled=true;
				SetAbort();
			}
		}

		if (m_Abort) return 1;
	}
	while (response!="1");

	m_SegmentsStartedExecuting = false;

	return 0;
}

int CCoordMotion::WaitForMoveXYZAFinished()
{
	CString response;

	int count=0;
	do
	{
		if (count++) Sleep(10);
		if (KMotionDLL.WriteLineReadLine(m_board,"CheckDoneXYZA",response.GetBufferSetLength(MAX_LINE))) {SetAbort(); return 1;}
		response.ReleaseBuffer();

		if (response == "-1")
		{
			m_AxisDisabled=true;
			SetAbort();
		}
		
		if (m_Abort) return 1;
	}
	while (response!="1");

	return 0;
}


// execute whatever segments are in the queue

int CCoordMotion::FlushSegments()
{
	int ispecial_cmd=0;

	// there are no more segments in this path, so maximize them
	// since we were waiting to see in there might have been
	// another one to combine
	MaximizeSegments();

	if (m_Simulate) return 0;  // exit if we are simulating

	// now download the 3 trip states for the segment

	if (nsegs>0 || nspecial_cmds>0)
	{
		if (m_nsegs_downloaded==0)
		{
			// if the first one, open it

			if (WaitForSegmentsFinished()) {SetAbort(); return 1;}

			if(KMotionDLL.WriteLine(m_board,"OpenBuf")) {SetAbort(); return 1;}
		}

		CString s;
	
		int iseg;
		for (iseg=m_nsegs_downloaded; iseg<nsegs; iseg++)
		{
			// see if there are any special commands that need to be inserted before segs

			if (DoSpecialCommand(iseg)){SetAbort(); return 1;} ;
			if (OutputSegment(iseg))   {SetAbort(); return 1;}  // Output the Segment
		}
			
		// see if there are any special commands that need to be inserted after segs

		if (DoSpecialCommand(iseg)) {SetAbort(); return 1;} ;

		// Execute segments and special commands

		if (!m_SegmentsStartedExecuting)
		{
			if(KMotionDLL.WriteLine(m_board,"ExecBuf")){SetAbort(); return 1;}
			m_SegmentsStartedExecuting = true;
		}
		tp_init(Kinematics->m_MotionParams.CollinearTol);
		DownloadInit();
	}
	return 0;
}


// intitalize everything related to downloading and look ahead times


void CCoordMotion::DownloadInit()
{
	m_nsegs_downloaded=0;
	m_TotalDownloadedTime = m_TimeAlreadyExecuted = 0.0;
}



// Called when the Interpreter is stopping due to either
// a single step, a halt, or program stop
//
// if some segments have been downloaded and execution
// is in progress, then we should send whatever is
// left in the TP which should bring everything to a 
// safe stop similar as if we had a break angle at
// this point
//
// 

int CCoordMotion::ExecutionStop()
{
	return FlushSegments();
}



int CCoordMotion::DoSpecialCommand(int iseg)
{
	// see if there are any special commands that need to be inserted before segs

	while (ispecial_cmd_downloaded < nspecial_cmds &&
		   iseg == special_cmds[ispecial_cmd_downloaded].seg_before)
	{
		if (KMotionDLL.WriteLine(m_board,special_cmds[ispecial_cmd_downloaded].cmd)) return 1;
		
		if (m_Abort) return 1;
		
		ispecial_cmd_downloaded++;
	}
	return 0;
}


int CCoordMotion::OutputSegment(int iseg)
{
	CString s;

	// output the segments

	tp_calc_seg_trip_states(iseg);

	for (int i=0; i<3; i++)
	{
		if (segments[iseg].C[i].t > 0.0)  // discard zero time segments
		{
			if (segments[iseg].type == SEG_LINEAR)
			{
				double dx = segments[iseg].x1-segments[iseg].x0;
				double dy = segments[iseg].y1-segments[iseg].y0;
				double dz = segments[iseg].z1-segments[iseg].z0;
				double da = segments[iseg].a1-segments[iseg].a0;
				
				double sum2,invd;
				
				sum2 = dx*dx + dy*dy + dz*dz + da*da;
				
				if (sum2>0.0f)
					invd = 1.0f/sqrt(sum2); // inverse total length
				else    
					invd = 0.0f;

				//  send as hex binary 32 bit floats


				float FloatArray[13];
				double Acts[MAX_ACTUATORS];
				
				if (Kinematics->TransformCADtoActuators(segments[iseg].x0,segments[iseg].y0,segments[iseg].z0,segments[iseg].a0,Acts)) return 1;

				FloatArray[0]  = (float)Acts[0];
				FloatArray[1]  = (float)Acts[1];
				FloatArray[2]  = (float)Acts[2];
				FloatArray[3]  = (float)Acts[3];

				if (Kinematics->TransformCADtoActuators(segments[iseg].x1,segments[iseg].y1,segments[iseg].z1,segments[iseg].a1,Acts)) return 1;
				
				FloatArray[4]  = (float)Acts[0];
				FloatArray[5]  = (float)Acts[1];
				FloatArray[6]  = (float)Acts[2];
				FloatArray[7]  = (float)Acts[3];

				FloatArray[8]  = 0.0f;
				FloatArray[9]  = (float)(segments[iseg].C[i].a*invd);
				FloatArray[10] = (float)(segments[iseg].C[i].b*invd);
				FloatArray[11] = (float)(segments[iseg].C[i].c*invd);
				FloatArray[12] = (float)(segments[iseg].C[i].t);

				int *Int = (int *)FloatArray;

				s.Format("LinearHex %X %X %X %X %X %X %X %X %X %X %X %X %X",
					Int[0],Int[1],Int[2],Int[3],Int[4],Int[5],Int[6],Int[7],Int[8],Int[9],Int[10],Int[11],Int[12]); 

			}
			else
			{
				double invd, radius, theta0, theta1, d_theta, dxy;
				double dx = segments[iseg].x0-segments[iseg].xc;
				double dy = segments[iseg].y0-segments[iseg].yc;
				double dz = segments[iseg].z1-segments[iseg].z0;
				double sum2;
				
				radius = sqrt(dx*dx + dy*dy); 

				theta0 = atan2(segments[iseg].y0-segments[iseg].yc,segments[iseg].x0-segments[iseg].xc);
				theta1 = atan2(segments[iseg].y1-segments[iseg].yc,segments[iseg].x1-segments[iseg].xc);  
									
				d_theta =  theta1 - theta0;        

				if (segments[iseg].DirIsCCW)
				{
					if (d_theta < 0.0f) d_theta+=TWO_PI; // CCW delta should be +  
				}
				else
				{
					if (d_theta > 0.0f) d_theta-=TWO_PI;  // CW delta should be -
				}
				
				dxy = d_theta * radius;  // path length along circle                                   
				
				sum2 = dxy*dxy + dz*dz;
									  
				if (sum2>0.0f)                      
					invd = 1.0f/sqrt(sum2); // inverse total length
				else
					sum2=0.0f;
				
				float FloatArray[13];


				double Acts[MAX_ACTUATORS];
				
				if (Kinematics->TransformCADtoActuators(segments[iseg].xc,segments[iseg].yc,segments[iseg].z0,0,Acts)) return 1;

				FloatArray[0]  = (float)Acts[0];
				FloatArray[1]  = (float)Acts[1];

				FloatArray[2]  = (float)(radius * Kinematics->m_MotionParams.CountsPerInchX);
				FloatArray[3]  = (float)(radius * Kinematics->m_MotionParams.CountsPerInchY);
				
				FloatArray[4]  = (float)(theta0);
				FloatArray[5]  = (float)(d_theta);
				FloatArray[6]  = (float)Acts[2];

				if (Kinematics->TransformCADtoActuators(segments[iseg].xc,segments[iseg].yc,segments[iseg].z1,0,Acts)) return 1;
				
				FloatArray[7]  = (float)Acts[2];
				FloatArray[8]  = (float)(0.0);
				FloatArray[9]  = (float)(segments[iseg].C[i].a * invd);
				FloatArray[10] = (float)(segments[iseg].C[i].b * invd);
				FloatArray[11] = (float)(segments[iseg].C[i].c * invd);
				FloatArray[12] = (float)(segments[iseg].C[i].t);

				int *Int = (int *)FloatArray;

				s.Format("ArcHex %X %X %X %X %X %X %X %X %X %X %X %X %X",
					Int[0],Int[1],Int[2],Int[3],Int[4],Int[5],Int[6],Int[7],Int[8],Int[9],Int[10],Int[11],Int[12]); 

			}
		
			if (KMotionDLL.WriteLine(m_board,s)) return 1;
			if (m_Abort) return 1;
			
			m_TotalDownloadedTime += segments[iseg].C[i].t;  // sum all downloaded times

			if (!m_SegmentsStartedExecuting && (m_TotalDownloadedTime  - m_TimeAlreadyExecuted) > Kinematics->m_MotionParams.TPLookahead)
			{
				if (KMotionDLL.WriteLine(m_board,"ExecBuf")) return 1;
				m_SegmentsStartedExecuting = true;
			}

			if (m_SegmentsStartedExecuting)
			{
				CString response;
				bool wait;
				int result;

				// Check how far ahead we are

				// first check old value of m_TimeAlreadyExecuted, if we aren't too far ahead of that
				// then don't bother getting new value from DSP

				wait = (m_TotalDownloadedTime - m_TimeAlreadyExecuted) > Kinematics->m_MotionParams.TPLookahead;
					
				while (wait && m_SegmentsStartedExecuting)
				{
					if (KMotionDLL.WriteLineReadLine(m_board,"ExecTime",response.GetBufferSetLength(MAX_LINE))) return 1;
					response.ReleaseBuffer();
					result=sscanf(response, "%lf",&m_TimeAlreadyExecuted);
					if (result != 1)  return 1;

					if (m_TimeAlreadyExecuted < 0.0)
					{
						result = AfxMessageBox("Unexpected Coordinated Motion Buffer Underflow!\r"
							"(Consider increasing the Trajectory Planner Lookahead time in the Tools Setup Configuration Screen)\r\r"
							"Press OK to attempt to continue motion\r"
							"Press CANCEL to abort",
							MB_ICONEXCLAMATION|MB_OKCANCEL);   

						if (result == IDCANCEL)
						{
							SetAbort();
							return 1;
						}
						else
						{
							m_SegmentsStartedExecuting = false;
						}
					}
					else
					{
						wait = (m_TotalDownloadedTime - m_TimeAlreadyExecuted) > Kinematics->m_MotionParams.TPLookahead;

						if (wait) Sleep(100);

						if (m_Abort) return 1;
					}
				}
			}
		}
	}
	m_nsegs_downloaded++;
	return 0;
}


// download any "Done" segments
//
// "Done" segments are those that are already max'ed out based on constraints
// other than the length of the vector, so adding/considering more vectors will
// not affect them.

int CCoordMotion::DownloadDoneSegments()
{
	int ispecial_cmd=0;

	if (m_Simulate) return 0;  // exit if we are simulating

	if (nsegs > m_nsegs_downloaded)
	{
		// check if we have at least one more that is "done"

		if (segments[m_nsegs_downloaded].Done)
		{
			// yes, we have at least one to download

			if (m_nsegs_downloaded==0)
			{
				// if the first one, wait for the buffer and open it

				if (WaitForSegmentsFinished()) {SetAbort(); return 1;}

				if(KMotionDLL.WriteLine(m_board,"OpenBuf")) {SetAbort(); return 1;}
				m_SegmentsStartedExecuting = false;
			}

			for (int iseg=m_nsegs_downloaded; iseg<nsegs && segments[iseg].Done; iseg++)
			{
				if (OutputSegment(iseg)){SetAbort(); return 1;}  // Output the Segment
			}
		}
	}
	return 0;
}





int CCoordMotion::GetAxisDefinitions(int *x, int *y, int *z, int *a)
{
	CString response;
	int result;

	if (!m_DefineCS_valid)
	{
		if (KMotionDLL.WriteLineReadLine(m_board,"DefineCS",response.GetBufferSetLength(MAX_LINE))) return 1;
		response.ReleaseBuffer();
		result=sscanf(response, "%d%d%d%d",&x_axis,&y_axis,&z_axis,&a_axis);
		if (result != 4) return 1;
	}

	*x = x_axis;
	*y = y_axis;
	*z = z_axis;
	*a = a_axis;

	return 0;
}

int CCoordMotion::ReadAndSyncCurPositions()
{
	// find out which axis is which

	// force refresh and save resuts

	m_DefineCS_valid = false;

	// don't sample positions until everything is stopped 
	if (WaitForSegmentsFinished()) {SetAbort(); return 1;}
	if (WaitForMoveXYZAFinished()) {SetAbort(); return 1;}

	if (ReadCurAbsPosition(&current_x,&current_y,&current_z,&current_a)) return 1;

	// convert absolute coordinates to GCode interpreter coordinates

	ConvertAbsoluteToInterpreterCoord(current_x,current_y,current_z,current_a,
		&_setup.current_x,&_setup.current_y,&_setup.current_z,&_setup.AA_current);

	return 0;    
}



int CCoordMotion::ReadCurAbsPosition(double *x, double *y, double *z, double *a)
{
	double tx,ty,tz,ta;

	// find out which axis is which

	if (GetAxisDefinitions(&x_axis,&y_axis,&z_axis,&a_axis)) {SetAbort(); return 1;}

	// read and set all axis (if undefined return interpreter)

	double Acts[MAX_ACTUATORS];

	for (int i=0; i<MAX_ACTUATORS; i++) Acts[i]=0.0;

	if (x_axis >=0)	if (GetDestination(x_axis,&Acts[0])) {SetAbort(); return 1;}
	if (y_axis >=0)	if (GetDestination(y_axis,&Acts[1])) {SetAbort(); return 1;}
	if (z_axis >=0)	if (GetDestination(z_axis,&Acts[2])) {SetAbort(); return 1;}
	if (a_axis >=0)	if (GetDestination(a_axis,&Acts[3])) {SetAbort(); return 1;}

	Kinematics->TransformActuatorstoCAD(Acts,&tx,&ty,&tz,&ta);

	if (x_axis < 0)	*x = current_x; else *x = tx;
	if (y_axis < 0) *y = current_y; else *y = ty;
	if (z_axis < 0)	*z = current_z; else *z = tz;
	if (a_axis < 0) *a = current_a; else *a = ta;

	return 0;    
}



void CCoordMotion::ConvertAbsoluteToInterpreterCoord(double x,double y,double z,double a, 
													double *gx,double *gy,double *gz,double *ga)
{
	*gx  = GC->InchesToUserUnits(x) - _setup.axis_offset_x  - _setup.origin_offset_x;
	*gy  = GC->InchesToUserUnits(y) - _setup.axis_offset_y  - _setup.origin_offset_y;
	*gz  = GC->InchesToUserUnits(z) - _setup.axis_offset_z  - _setup.origin_offset_z;
	*ga  = GC->InchesToUserUnits(a) - _setup.AA_axis_offset - _setup.AA_origin_offset;
}

void CCoordMotion::ConvertAbsoluteToMachine(double x,double y,double z,double a, 
											double *gx,double *gy,double *gz,double *ga)
{
	*gx  = GC->InchesToUserUnits(x);
	*gy  = GC->InchesToUserUnits(y);
	*gz  = GC->InchesToUserUnits(z);
	*ga  = GC->InchesToUserUnits(a);
}

int CCoordMotion::GetDestination(int axis, double *d)
{
	int result;
	CString cmd,response;

	*d=0.0;

	if (axis==-1) return 0;  // not used in coordinate system 
	
	if (axis<0 || axis>N_CHANNELS) {SetAbort(); return 1;} // invalid

	cmd.Format("Dest%d",axis);
	if (KMotionDLL.WriteLineReadLine(m_board,cmd,response.GetBufferSetLength(MAX_LINE))) {SetAbort(); return 1;}
	response.ReleaseBuffer();

	result=sscanf(response, "%lf",d);
	if (result != 1) {SetAbort(); return 1;}

	return 0;
}

void CCoordMotion::SetFeedRateOverride(double v)
{
	m_FeedRateOverride=v;
}

double CCoordMotion::GetFeedRateOverride()
{
	return m_FeedRateOverride;
}

void CCoordMotion::SetStraightTraverseCallback(STRAIGHT_TRAVERSE_CALLBACK *p)
{
	m_StraightTraverseCallback=p;
}

void CCoordMotion::SetStraightFeedCallback(STRAIGHT_FEED_CALLBACK *p)
{
	m_StraightFeedCallback=p;
}

void CCoordMotion::SetArcFeedCallback(ARC_FEED_CALLBACK *p)
{
	m_ArcFeedCallback=p;
}


void CCoordMotion::SetAbort()
{
	m_Abort=true;
}

void CCoordMotion::ClearAbort()
{
	if (m_Abort)
	{
		// If we had been aborted then
		// initialize the trajectory planner
		tp_init(Kinematics->m_MotionParams.CollinearTol);	 
		DownloadInit();  // intialize download/look ahead variables  
	}
	m_Abort=false;
}

bool CCoordMotion::GetAbort()
{
	return m_Abort;
}


int CCoordMotion::MeasurePointAppendToFile(const char *name)
{
	double x,y,z,a;
	static int row=0, col=0;
	int NRows,NCols,r,c;
	double GeoSpacingX,GeoSpacingY,X,Y;

	// first beg of file to see how many rows and cols
	// and how much data is there, if none then assume
	// starting over

	FILE *f = fopen(name,"rt");

	if (!f)
	{
		AfxMessageBox((CString)"Unable to open Geometric Correction File : " + name);
		return 1;
	}

	int result = fscanf(f,"%d,%d",&NRows,&NCols);
		
	if (result != 2 || NRows < 2 || NRows > 1000 || NCols < 2 || NCols > 1000)
	{
		fclose(f);
		AfxMessageBox((CString)"Invalid Geometric Correction File (NRows and NCols) : " + name);
		return 1;
	}

	result = fscanf(f,"%lf,%lf",&GeoSpacingX,&GeoSpacingY);
		
	if (result != 2)
	{
		fclose(f);
		AfxMessageBox((CString)"Invalid Geometric Correction File (GeoSpacingX and GeoSpacingY) : " + name);
		return 1;
	}

	result = fscanf(f,"%d,%d,%lf,%lf",&r,&c,&X,&Y);

	if (result != 4) row=col=0;		// assume we are starting over

	fclose(f);


	if (ReadCurAbsPosition(&x,&y,&z,&a)) return 1;
	
	f = fopen(name,"at");

	if (!f)
	{
		AfxMessageBox((CString)"Unable to open Measurement Point File : " + name);
		return 1;
	}

	fprintf(f,"%d,%d,%f,%f,%f\n",row,col,x,y,z);

	col++;

	if (col == NCols)
	{
		col = 0;
		row++;
	}

	fclose(f);

	return 0;
}

